import { supabase } from '@/config/supabase';
import { v4 as uuidv4 } from 'uuid';
import {
  Invoice,
  InvoiceItem,
  Transaction,
  Payout,
  CreateInvoiceRequest,
  InvoiceStatus,
  Currency
} from './types';

// Helper to get system fee percent (simulate for now)
async function getFeePercent() {
  // In real app, fetch from system_settings
  return 2.5;
}

export const invoiceService = {
  // Create invoice
  async createInvoice(vendor_id: string, req: CreateInvoiceRequest) {
    // 1. Calculate subtotal, fee, total
    const subtotal = req.items.reduce((sum, item) => sum + item.quantity * item.unit_price, 0);
    const feePercent = await getFeePercent();
    const fee = Math.round((subtotal * feePercent) / 100);
    const total = subtotal + fee;
    // 2. Generate public code
    const public_code = uuidv4().slice(0, 10);
    // 3. Insert invoice
    const { data: invoice, error: invoiceError } = await supabase.from('invoices').insert({
      vendor_id,
      customer_id: req.customer_id || null,
      customer_name: req.customer_name,
      customer_email: req.customer_email,
      currency: req.currency,
      subtotal,
      fee,
      total,
      status: 'pending',
      notes: req.notes,
      due_date: req.due_date,
      public_code,
      pdf_url: null // Placeholder, to be updated after PDF generation
    }).select().single();
    if (invoiceError) return { error: invoiceError };
    // 4. Insert invoice items
    const itemsToInsert = req.items.map(item => ({
      invoice_id: invoice.id,
      name: item.name,
      quantity: item.quantity,
      unit_price: item.unit_price,
      total_price: item.quantity * item.unit_price
    }));
    const { error: itemsError } = await supabase.from('invoice_items').insert(itemsToInsert);
    if (itemsError) return { error: itemsError };
    // 5. (Future) Generate PDF and update pdf_url
    // ...
    return { data: { ...invoice, items: itemsToInsert, public_link: `/i/${public_code}` } };
  },

  // View invoice (public)
  async viewInvoice(public_code: string) {
    const { data: invoice, error } = await supabase.from('invoices').select('*,items:invoice_items(*),vendor:vendor_id(*)').eq('public_code', public_code).single();
    if (error) return { error };
    return { data: invoice };
  },

  // Mark invoice as paid (webhook)
  async markPaid(invoice_id: string, payment_gateway: string, amount: number, currency: Currency) {
    // 1. Update invoice status
    const { data: invoice, error: invoiceError } = await supabase.from('invoices').update({ status: 'paid' }).eq('id', invoice_id).select().single();
    if (invoiceError) return { error: invoiceError };
    // 2. Log transaction
    const { data: txn, error: txnError } = await supabase.from('transactions').insert({
      invoice_id,
      vendor_id: invoice.vendor_id,
      amount,
      currency,
      paid_at: new Date().toISOString(),
      payment_gateway,
      status: 'success'
    }).select().single();
    if (txnError) return { error: txnError };
    // 3. Trigger payout (pending)
    await supabase.from('payouts').insert({
      vendor_id: invoice.vendor_id,
      amount: amount - invoice.fee,
      status: 'pending'
    });
    return { data: { invoice, transaction: txn } };
  },

  // List invoices for vendor
  async listInvoices(vendor_id: string, filters: any) {
    let query = supabase.from('invoices').select('*,items:invoice_items(*)').eq('vendor_id', vendor_id);
    if (filters.status) query = query.eq('status', filters.status);
    if (filters.currency) query = query.eq('currency', filters.currency);
    if (filters.date_range) {
      const [from, to] = filters.date_range;
      query = query.gte('created_at', from).lte('created_at', to);
    }
    const { data, error } = await query.order('created_at', { ascending: false });
    return { data, error };
  },

  // Download PDF (returns URL)
  async getPdfUrl(invoice_id: string, vendor_id: string) {
    const { data: invoice, error } = await supabase.from('invoices').select('pdf_url').eq('id', invoice_id).eq('vendor_id', vendor_id).single();
    if (error) return { error };
    // Optionally regenerate PDF if missing
    return { data: invoice.pdf_url };
  },

  // Delete invoice (only if pending)
  async deleteInvoice(invoice_id: string, vendor_id: string) {
    // Only allow if status is pending
    const { data: invoice, error: fetchError } = await supabase.from('invoices').select('status').eq('id', invoice_id).eq('vendor_id', vendor_id).single();
    if (fetchError) return { error: fetchError };
    if (invoice.status !== 'pending') return { error: { message: 'Only pending invoices can be deleted.' } };
    const { error } = await supabase.from('invoices').delete().eq('id', invoice_id).eq('vendor_id', vendor_id);
    return { success: !error, error };
  },

  // List transactions for vendor
  async listTransactions(vendor_id: string) {
    const { data, error } = await supabase.from('transactions').select('*').eq('vendor_id', vendor_id).eq('status', 'success').order('paid_at', { ascending: false });
    return { data, error };
  }
}; 